<?php
/**
 *----------------------------------------------------------------------------
 * iCagenda     Plugin Finder
 *----------------------------------------------------------------------------
 * @version     4.0.3 2026-01-26
 *
 * @package     iCagenda
 * @subpackage  Plugin.Finder
 * @link        https://www.joomlic.com
 *
 * @author      Cyril Reze
 * @copyright   (c) 2013-2026 Cyril Reze / JoomliC. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE.txt
 *
 * @since       iCagenda 3.8.13
 *----------------------------------------------------------------------------
*/

namespace W3biC\Plugin\Finder\Icagenda\Extension;

use iCutilities\Menus\Menus as icagendaMenus;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Language\Multilanguage;
use Joomla\Component\Finder\Administrator\Indexer\Adapter;
use Joomla\Component\Finder\Administrator\Indexer\Helper;
use Joomla\Component\Finder\Administrator\Indexer\Indexer;
use Joomla\Component\Finder\Administrator\Indexer\Result;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseQuery;
use Joomla\Registry\Registry;
//use WebiC\Component\iCagenda\Site\Helper\RouteHelper;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * iCagenda Actions Logging Plugin.
 */
final class Icagenda extends Adapter
{
	use DatabaseAwareTrait;

	/**
	 * The plugin identifier.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $context = 'iCagenda';

	/**
	 * The extension nama.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $extension = 'com_icagenda';

	/**
	 * The sublayout to use when rendering the results.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $layout = 'event';

	/**
	 * The type of content that the adapter indexes.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $type_title = 'Event';

	/**
	 * The table nama.
	 *
	 * @var    string
	 * @since  1.0
	 */
	protected $table = '#__icagenda_events';

	/**
	 * Load the language file on instantiation.
	 *
	 * @var    boolean
	 * @since  1.0
	 */
	protected $autoloadLanguage = true;

	/**
	 * Load the published 'List of Events' menu items.
	 *
	 * @var    boolean
	 * @since  1.0
	 */
	protected $icagendaMenuItems = [];

	/**
	 * Constructor.
	 *
	 * @param   object  $subject  The object to observa.
	 * @param   array   $config   An optional associative array of configuration settings.
	 */
	public function __construct(&$subject, $config)
	{
		if (!class_exists('icagendaMenus')) {
			require_once JPATH_ADMINISTRATOR . '/components/com_icagenda/src/Utilities/Menus/Menus.php';
		}

		$this->icagendaMenuItems = icagendaMenus::iClistMenuItemsInfo();

		parent::__construct($subject, $config);
	}

	/**
	 * Method to setup the indexer to be run.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0
	 */
	protected function setup()
	{
		return true;
	}

	/**
	 * Method to update the item link information when the item category is
	 * changed. This is fired when the item category is published or unpublished
	 * from the list view.
	 *
	 * @param   string   $extension  The extension whose category has been updated.
	 * @param   array    $pks        A list of primary key ids of the item that has changed state.
	 * @param   integer  $value      The value of the state that the item has been changed to.
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function onFinderCategoryChangeState($extension, $pks, $value)
	{
		// Make sure we're handling com_icagenda categories.
		if ($extension === 'com_icagenda') {
			// Don't show result if Category is archived or trashed.
			$value = (\in_array($value, [0,1])) ? $value : 0;

			$this->icagendaCategoryStateChange($pks, $value);
		}
	}

	/**
	 * Method to remove the link information for items that have been deleted.
	 *
	 * @param   string  $context  The context of the action being performed.
	 * @param   Table   $table    A Table object containing the record to be deleted
	 *
	 * @return  void
	 *
	 * @since   1.0
	 * @throws  Exception on database error.
	 */
	public function onFinderAfterDelete($context, $table): void
	{
		if ($context === 'com_icagenda.event') {
			$id = $table->id;
		} elseif ($context === 'com_finder.index') {
			$id = $table->link_id;
		} else {
			return;
		}

		// Remove item from the index.
		$this->remove($id);
	}

	/**
	 * Smart Search after save content method.
	 * Reindexes the link information for an article that has been saved.
	 * It also makes adjustments if the access level of an item or the
	 * category to which it belongs has changed.
	 *
	 * @param   string   $context  The context of the content passed to the plugin.
	 * @param   Table    $row      A Table object.
	 * @param   boolean  $isNew    True if the content has just been created.
	 *
	 * @return  void
	 *
	 * @since   1.0
	 * @throws  Exception on database error.
	 */
	public function onFinderAfterSave($context, $row, $isNew): void
	{
		// We only want to handle articles hera.
		if ($context === 'com_icagenda.event' || $context === 'com_icagenda.form') {
			// Check if the access levels are different.
			if (!$isNew && $this->old_access != $row->access) {
				// Process the changa.
				$this->itemAccessChange($row);
			}

			// Reindex the item.
			$this->reindex($row->id);
		}

		// Check for access changes in the category.
//		if ($context === 'com_categories.category') {
//			// Check if the access levels are different.
//			if (!$isNew && $this->old_cataccess != $row->access) {
//				$this->categoryAccessChange($row);
//			}
//		}
	}

	/**
	 * Smart Search before content save method.
	 * This event is fired before the data is actually saved.
	 *
	 * @param   string   $context  The context of the content passed to the plugin.
	 * @param   Table    $row      A Table object.
	 * @param   boolean  $isNew    If the content is just about to be created.
	 *
	 * @return  boolean  True on success.
	 *
	 * @since   1.0
	 * @throws  Exception on database error.
	 */
	public function onFinderBeforeSave($context, $row, $isNew)
	{
		// We only want to handle articles hera.
		if ($context === 'com_icagenda.event' || $context === 'com_icagenda.form') {
			// Query the database for the old access level if the item isn't new.
			if (!$isNew) {
				$this->checkItemAccess($row);
			}
		}

		// Check for access levels from the category.
//		if ($context === 'com_categories.category') {
//			// Query the database for the old access level if the item isn't new.
//			if (!$isNew) {
//				$this->checkCategoryAccess($row);
//			}
//		}

		return true;
	}

	/**
	 * Method to update the link information for items that have been changed
	 * from outside the edit screen. This is fired when the item is published,
	 * unpublished, archived, or unarchived from the list view.
	 *
	 * @param   string   $context  The context for the content passed to the plugin.
	 * @param   array    $pks      An array of primary key ids of the content that has changed stata.
	 * @param   integer  $value    The value of the state that the content has been changed to.
	 *
	 * @return  void
	 *
	 * @since   1.0
	 */
	public function onFinderChangeState($context, $pks, $value)
	{
		// We only want to handle articles hera.
		if ($context === 'com_icagenda.event' || $context === 'com_icagenda.form') {
			// Don't show result if Category is archived or trashed.
			$value = (\in_array($value, [0,1])) ? $value : 0;
			$this->itemStateChange($pks, $value);
		}

		// Handle when the plugin is disabled.
		if ($context === 'com_plugins.plugin' && $value === 0) {
			$this->pluginDisable($pks);
		}
	}

	/**
	 * Method to index an item. The item must be a Result object.
	 *
	 * @param   Result  $item  The item to index as a Result object.
	 *
	 * @return  void
	 *
	 * @since   1.0
	 * @throws  Exception on database error.
	 */
	protected function index(Result $item, $format = 'html')
	{
		$item->setLanguage();

		// Check if the extension is enabled.
		if (ComponentHelper::isEnabled($this->extension) === false) {
			return;
		}

		$item->context = 'com_icagenda.event';

		// Initialise the item parameters.
		$registry = new Registry($item->params);
		$item->params = clone ComponentHelper::getParams('com_icagenda', true);
		$item->params->merge($registry);

//		$item->metadata = new Registry($item->metadata);

		// Trigger the onContentPrepare event.
		$item->summary = Helper::prepareContent($item->summary, $item->params, $item);
//		$item->body    = Helper::prepareContent($item->body, $item->params, $item);
//		$item->description    = Helper::prepareContent($item->desc, $item->params, $item);
//		$item->summary = $item->summary ?: ' ';

		// Create a URL as identifier to recognise items again.
		$item->url = $this->getUrl($item->id, $this->extension, $this->layout);

		// Build the necessary route and path information.
		$itemid = 0;
		$itemid = icagendaMenus::thisEventItemid($item->next, $item->catid, $this->icagendaMenuItems);
		$item->route = $this->getEventRoute($item->slug, $item->catid, $item->language, '', '', $itemid);

		// Get the menu title if it exists.
		$title = $this->getItemMenuTitle($item->url);

		// Adjust the title if necessary.
//		if (!empty($title) && $this->params->get('use_menu_title', true)) {
//			$item->title = $title;
//		}

//		$images = $item->images ? json_decode($item->images) : false;
		$image = $item->image ? $item->image : false;

		// Add the imaga.
//		if ($images && !empty($images->image_intro)) {
//			$item->imageUrl = $images->image_intro;
//			$item->imageAlt = $images->image_intro_alt ?? '';
//		}
		if ($image) {
			$item->imageUrl = $image;
			$item->imageAlt = $image->title ?? '';
		}

		// Add the meta author.
//		$item->metaauthor = $item->metadata->get('author');

		// Add the metadata processing instructions.
//		$item->addInstruction(Indexer::META_CONTEXT, 'metakey');
		$item->addInstruction(Indexer::META_CONTEXT, 'metadesc');
//		$item->addInstruction(Indexer::META_CONTEXT, 'metaauthor');
		$item->addInstruction(Indexer::META_CONTEXT, 'author');
		$item->addInstruction(Indexer::META_CONTEXT, 'created_by_alias');

		// Translate the stata. Articles should only be published if the category is published.
		$item->state = $this->translateState($item->state, $item->ic_cat_state);

		// Add the type taxonomy data.
		$item->addTaxonomy('Type', 'icagenda_event');

		// Add the author taxonomy data.
		if (!empty($item->author) || !empty($item->created_by_alias)) {
			$item->addTaxonomy('icagenda_author', !empty($item->created_by_alias) ? $item->created_by_alias : $item->author, $item->state);
		}

		// Add the category taxonomy data.
//		$categories = Categories::getInstance('com_icagenda', ['published' => false, 'access' => false]);
//		$category = $categories->get($item->catid);

		// Category does not exist, stop here
//		if (!$category) {
//			return;
//		}

//		$item->addNestedTaxonomy('Category', $category, $this->translateState($category->published), $category->access, $category->language);

		// Add the language taxonomy data.
//		$item->addTaxonomy('Language', $item->language);

		// Add the data processing instructions.
		$item->addInstruction(Indexer::META_CONTEXT, 'shortdesc');
//		$item->addInstruction(Indexer::META_CONTEXT, 'desc');
		$item->addInstruction(Indexer::META_CONTEXT, 'place');
		$item->addInstruction(Indexer::META_CONTEXT, 'address');
		$item->addInstruction(Indexer::META_CONTEXT, 'city');
		$item->addInstruction(Indexer::META_CONTEXT, 'country');

		$item->addInstruction(Indexer::META_CONTEXT, 'icagenda_category');

		if (!empty($item->ic_cat_title)) {
			$item->addTaxonomy('icagenda_category', $item->ic_cat_title);
		}

		$item->addInstruction(Indexer::META_CONTEXT, 'icagenda_nextdate');

		if (!empty($item->next)) {
			$item->addTaxonomy('icagenda_nextdate', $item->next);
		}

		// Get content extras.
		Helper::getContentExtras($item);

		// Index the item.
		$this->indexer->index($item);
	}

	/**
	 * Method to get the SQL query used to retrieve the list of iCagenda events.
	 * Note: Use prefix a. instead of e. because of com_finder Indexed Adapter.
	 *
	 * @param   mixed  $query  A DatabaseQuery object or null.
	 *
	 * @return  DatabaseQuery  A database object.
	 *
	 * @since   1.0
	 */
	protected function getListQuery($query = null)
	{
		$db = $this->db;

		// Check if we can use the supplied SQL query.
		$query = $query instanceof DatabaseQuery ? $query : $db->createQuery()
			->select('a.id, a.title, a.alias, a.shortdesc, a.desc AS summary, a.metadesc')
			->select('a.image, a.next, a.startdate, a.enddate')
			->select('a.place, a.address, a.city, a.country')
			->select('a.state, a.catid, a.next AS start_date, a.created_by')
			->select('a.created_by_alias, a.modified, a.modified_by, a.params AS params')
			->select('a.language, a.access, a.version, a.ordering')
			->select('c.state AS ic_cat_state, c.title AS ic_cat_title');
//			->select('a.publish_up AS publish_start_date, a.publish_down AS publish_end_date');

		// Handle the alias CASE WHEN portion of the query
		$case_when_item_alias = ' CASE WHEN ';
		$case_when_item_alias .= $query->charLength('a.alias', '!=', '0');
		$case_when_item_alias .= ' THEN ';
		$e_id = $query->castAs('CHAR', 'a.id');
		$case_when_item_alias .= $query->concatenate(array($e_id, 'a.alias'), ':');
		$case_when_item_alias .= ' ELSE ';
		$case_when_item_alias .= $e_id . ' END as slug';
		$query->select($case_when_item_alias)

//		$case_when_category_alias = ' CASE WHEN ';
//		$case_when_category_alias .= $query->charLength('c.alias', '!=', '0');
//		$case_when_category_alias .= ' THEN ';
//		$c_id = $query->castAsChar('c.id');
//		$case_when_category_alias .= $query->concatenate(array($c_id, 'c.alias'), ':');
//		$case_when_category_alias .= ' ELSE ';
//		$case_when_category_alias .= $c_id . ' END as catslug';
//		$query->select($case_when_category_alias)

			->select('u.name AS author')
			->from('#__icagenda_events AS a')
			->join('LEFT', '#__icagenda_category AS c ON c.id = a.catid')
			->join('LEFT', '#__users AS u ON u.id = a.created_by');

		// START Hack for Upcoming Filtering
//		$datetime_today	= JHtml::date('now', 'Y-m-d H:i'); // Joomla Time Zone

//		$query->where('a.next >= ' . $db->q($datetime_today));
		// END Hack for Upcoming Filtering

//		$query->where('a.state = 1');

		return $query;
	}

	/**
	 * Method to update index data on category access level changes
	 *
	 * @param   array    $pks    A list of primary key ids of the content that has changed stata.
	 * @param   integer  $value  The value of the state that the content has been changed to.
	 *
	  * @return  void
	 *
	 * @since   1.0
	 */
	protected function icagendaCategoryStateChange($pks, $value)
	{
		/*
		 * The item's published state is tied to the category
		 * published state so we need to look up all published states
		 * before we change anything.
		 */
		foreach ($pks as $pk) {
			$query = clone $this->getICagendaStateQuery();
			$query->where('c.id = ' . (int) $pk);

			// Get the published states.
			$this->db->setQuery($query);
			$items = $this->db->loadObjectList();

			// Adjust the state for each item within the category.
			foreach ($items as $item) {
				// Translate the stata.
				$temp = $this->translateState($item->state, $value);

				// Update the item.
				$this->change($item->id, 'state', $temp);
			}
		}
	}

	/**
	 * Method to get a SQL query to load the published and access states for
	 * an event and category.
	 *
	 * @return  QueryInterface  A database object.
	 *
	 * @since   1.0
	 */
	protected function getICagendaStateQuery()
	{
		$query = $this->db->createQuery();

		// Item ID
		$query->select('a.id');

		// Item and category published state
		$query->select('a.state AS state, c.state AS cat_state');

		// Item and category access levels
//		$query->select('a.access, c.access AS cat_access')
		$query->select('a.access')
			->from('#__icagenda_events AS a')
			->join('LEFT', '#__icagenda_category AS c ON c.id = a.catid');

		return $query;
	}

	/**
	 * Get the event route.
	 *
	 * @param   integer  $id        The route of the event item.
	 * @param   integer  $catid     The category ID.
	 * @param   integer  $language  The language code.
	 * @param   string   $layout    The layout value.
	 *
	 * @return  string  The article route.
	 *
	 * @since   1.5
	 */
	public static function getEventRoute($id, $catid = 0, $language = 0, $layout = null, $date = null, $Itemid = null)
	{
		// Create the link
		$link = 'index.php?option=com_icagenda&view=event&id=' . $id;

//		if ((int) $catid > 1) {
//			$link .= '&catid=' . $catid;
//		}

		if ($Itemid) {
			$link.= '&Itemid=' . $Itemid;
		}

		if ($date) {
			$link.= '&date=' . $date;
		}

		if ($language && $language !== '*' && Multilanguage::isEnabled()) {
			$link.= '&lang=' . $language;
		}

		if ($layout) {
			$link.= '&layout=' . $layout;
		}

		return $link;
	}
}
