<?php
/**
 *----------------------------------------------------------------------------
 * iCagenda     Events Management Extension for Joomla!
 *              Module iCagenda Calendar
 *----------------------------------------------------------------------------
 * @version     4.0.0 2026-01-12
 *
 * @package     iCagenda.Site
 * @subpackage  mod_icagenda_calendar
 * @link        https://www.joomlic.com
 *
 * @author      Cyril Reze
 * @copyright   (c) 2012-2026 Cyril Reze / JoomliC. All rights reserved.
 * @license     GNU General Public License version 3 or later; see LICENSE.txt
 *
 * @since       4.0.0
 *----------------------------------------------------------------------------
 */

namespace WebiC\Module\IcagendaCalendar\Site\Helper;

use iClib\Color\Color as iCColor;
use iClib\Date\Date as iCDate;
use iClib\Date\Period as iCDatePeriod;
use iClib\Filter\Output as iCFilterOutput;
use iClib\Globalize\Convert as iCGlobalizeConvert;
use iClib\Globalize\Globalize as iCGlobalize;
use iClib\Library\Library as iCLibrary;
use iClib\String\StringHelper as iCString;
use iCutilities\Event\Event as icagendaEvent;
use iCutilities\Events\Events as icagendaEvents;
use iCutilities\Events\EventsData as icagendaEventsData;
use iCutilities\Media\Media as icagendaMedia;
use iCutilities\Menus\Menus as icagendaMenus;
use iCutilities\Registration\Registration as icagendaRegistration;
use iCutilities\Thumb\Thumb as icagendaThumb;

use Joomla\CMS\Access\Access;
use Joomla\CMS\Application\SiteApplication;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Helper\ModuleHelper;
use Joomla\CMS\HTML\HTMLHelper;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Layout\FileLayout;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Uri\Uri;

//use Joomla\Component\Content\Site\Helper\RouteHelper;
use Joomla\Component\Content\Site\Model\ArticlesModel;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Database\DatabaseInterface;

use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;

use WebiC\Component\iCagenda\Site\Helper\RouteHelper;
use WebiC\Component\iCagenda\Site\Model\EventsModel;

// phpcs:disable PSR1.Files.SideEffects
\defined('_JEXEC') or die;
// phpcs:enable PSR1.Files.SideEffects

/**
 * Helper for mod_icagenda_calendar
 */
class IcagendaCalendarHelper implements DatabaseAwareInterface
{
	use DatabaseAwareTrait;

	/**
	 * Method to get calendar HTML month view with AJAX
	 *
	 * @return  void
	 */
	public function calendarAjax()
	{
		$input = Factory::getApplication()->getInput();

		$calendarHTML = self::getCalendar(
			$input->get('modid', ''),
			$input->getString('params', ''),
			$input->get('year', date('Y')),
			$input->get('month', date('m'))
		);

		return (string) $calendarHTML;
	}

	/**
	 * Method to get HTML for events of the day with AJAX
	 *
	 * @return  void
	 */
	public function eventsAjax()
	{
		$input = Factory::getApplication()->getInput();

		$modid = $input->get('modid', '');
		$params = $input->getString('params', '');
		$date = $input->get('date', date('Y-m-d'));

		if ($date) {
			$eventsHTML = self::getEvents($modid, $params, $date);
			try {
				return (string) $eventsHTML;
			} catch (\Exception $e) {
				// Catch an exception and return the response.
				// Send the response.
				static::sendResponse($e);
			}
		}
	}

	/**
	 * Generate event calendar in HTML format
	 */
	public static function getCalendar($modid, $params, $year = '', $month = '')
	{
//		$starttime_cal = iCLibrary::getMicrotime();

		// Module Params
//		$modid  = $modid ?: $this->module->id;
		$module = ModuleHelper::getModuleById((string) $modid);

		$params = $params ?: $module->params;
//		$params = $module->params;

		$moduleParams = new Registry($params);
//		$moduleParams->loadString($params);

		$todayTZ = $moduleParams->get('today_timezone', '');

		if ($todayTZ == 'USER') {
			$tz = true;
		} elseif ($todayTZ == 'SITE') {
			$tz = false;
		} else {
			$tz = null;
		}

//		$todayUser = HTMLHelper::date('now', 'Y-m-d H:i', true);
//		echo 'Today User: ' . $todayUser . '<br>';
//		$todaySite = HTMLHelper::date('now', 'Y-m-d H:i', false);
//		echo 'Today Site: ' . $todaySite . '<br>';
//		$todayUTC = HTMLHelper::date('now', 'Y-m-d H:i', null);
//		echo 'Today UTC: ' . $todayUTC . '<br>';

		$monthTZ = HTMLHelper::date('now', 'm', $tz);
		$yearTZ  = HTMLHelper::date('now', 'Y', $tz);
		$todayTZ = HTMLHelper::date('now', 'Y-m-d', $tz);

		$initialDate = $moduleParams->get('initial_calendar_date', '');

		if (!$year && !$month && $initialDate) {
			$loadOnDate = $moduleParams->get('load_on_date', []);

			$year  = $loadOnDate->year ?: $yearTZ;
			$month = $loadOnDate->month ?: $monthTZ;
		}

		$calYear  = ($year != '') ? $year : $yearTZ;
		$calMonth = ($month != '') ? $month : $monthTZ;

		$calMonthFirstDay         = $calYear . '-' . $calMonth . '-01';
		$calMonthFirstDayPosition = date('N', strtotime($calMonthFirstDay));

		// Set the first day of the week (default: Monday).
		$firstDay = $moduleParams->get('first_day_of_week', 0);

		$calMonthFirstDayPosition = $calMonthFirstDayPosition + $firstDay;

		if ($calMonthFirstDayPosition > 7) {
			$calMonthFirstDayPosition = $calMonthFirstDayPosition - 7;
		}

		$fullDays = [
			'0' => Text::_('MONDAY'),
			'6' => Text::_('TUESDAY'),
			'5' => Text::_('WEDNESDAY'),
			'4' => Text::_('THURSDAY'),
			'3' => Text::_('FRIDAY'),
			'2' => Text::_('SATURDAY'),
			'1' => Text::_('SUNDAY'),
		];

		$shortDays = [
			'0' => Text::_('MON'),
			'6' => Text::_('TUE'),
			'5' => Text::_('WED'),
			'4' => Text::_('THU'),
			'3' => Text::_('FRI'),
			'2' => Text::_('SAT'),
			'1' => Text::_('SUN'),
		];

		$dayStyles = [
			'0' => '',
			'1' => ' ic-rounded-light',
			'2' => ' ic-rounded-medium',
			'3' => ' ic-rounded-strong',
			'4' => ' ic-circle',
		];

		$dayStyle = $moduleParams->get('day_style', 2);
		$dayClass = $dayStyles[$dayStyle];

		// Calculate the number of days to display.
		$calMonthNbOfDays        = cal_days_in_month(CAL_GREGORIAN, $calMonth, $calYear);
		$calMonthLastDayPosition = ($calMonthFirstDayPosition == 1)
			? ($calMonthNbOfDays)
			: ($calMonthNbOfDays + ($calMonthFirstDayPosition - 1));
		$calDisplay              = ($calMonthLastDayPosition <= 35) ? 35 : 42;

		$calYear_Prev  = date('Y', strtotime($calMonthFirstDay . ' - 1 Month'));
		$calMonth_Prev = date('m', strtotime($calMonthFirstDay . ' - 1 Month'));
		$calYear_Next  = date('Y', strtotime($calMonthFirstDay . ' + 1 Month'));
		$calMonth_Next = date('m', strtotime($calMonthFirstDay . ' + 1 Month'));

		$calMonthNbOfDays_Prev = cal_days_in_month(CAL_GREGORIAN, $calMonth_Prev, $calYear_Prev);

		$calDateStart = date('Y-m-d', strtotime($calMonthFirstDay . ' - ' . ($calMonthFirstDayPosition - 1) . ' DAYS'));
		$calDateEnd   = date('Y-m-d', strtotime($calDateStart . ' + ' . ($calDisplay + 1) . ' DAYS'));

		$calEvents = self::getEventsCurrentCal($modid, $params, $calDateStart, $calDateEnd);

		$dayCount    = 1;
		$currentWeek = '';
		$currentDate = $calYear . '-' . $calMonth . '-01';

		if (date('W', strtotime($currentDate)) == date('W')) {
			$currentWeek = ' current-week';
		}

		$thisMonthOnly = $moduleParams->get('this_month_only', 0);

		$dayA11y = $moduleParams->get('day_a11y', 0);
		$textSizeClass = ($dayA11y == 1 || $dayA11y == 2) ? ' ic-text-large' : '';

		$eventIndicator = $moduleParams->get('day_event_indicator', '');

		$dateClass = '';

		if ($eventIndicator) {
			$dateClass = ' ic-indicator';
		}

		$dialogStyle = '';
		$dialogExtraWidth = $moduleParams->get('dialog_extra_width', 0);

		if ($dialogExtraWidth) {
			$dialogStyle = ' style="width: calc(100% + ' . ($dialogExtraWidth * 2) . 'px); margin-left: -' . $dialogExtraWidth . 'px; max-width: 100vw;"';

//			echo '<style>
//				.ic-calendar-dialog {
//					width: calc(100% + ' . ($dialogExtraWidth * 2) . 'px) !important;
//					margin-left: -' . $dialogExtraWidth . 'px !important;
//					max-width: 100vw;
//				}
//			</style>';
		}
		?>

		<!-- iCagenda - Calendar -->
		<div class="ic-calendar-container">

			<dialog id="ic-calendar-dialog" class="ic-calendar-dialog"<?php echo $dialogStyle; ?>>
				<?php echo self::getEvents($modid, $params); ?>
			</dialog>

			<div class="ic-calendar-nav">
				<button
					class="ic-calendar-nav-prev"
					data-year="<?php echo $calYear_Prev; ?>"
					data-month="<?php echo $calMonth_Prev; ?>"
					onclick="JoomliC.changeMonth('<?php echo $modid; ?>', <?php echo htmlspecialchars($params); ?>, '<?php echo $calYear_Prev; ?>', '<?php echo $calMonth_Prev; ?>')">
				</button>
				<div class="ic-calendar-nav-selectors">
					<div class="ic-calendar-nav-month">
						<select class="ic-calendar-month-dropdown" onchange="JoomliC.updateCalendar()">
							<?php echo self::getMonthList(date('m',strtotime($calMonthFirstDay))); ?>
						</select>
					</div>
					<div class="ic-calendar-nav-year">
						<input
							type="number"
							class="ic-calendar-year-input"
							name="year"
							min="100"
							max="9999"
							value="<?php echo date('Y', strtotime($calMonthFirstDay)); ?>"
							onchange="JoomliC.updateCalendar()"
						/>
					</div>
				</div>
				<button
					class="ic-calendar-nav-next"
					data-year="<?php echo $calYear_Next; ?>"
					data-month="<?php echo $calMonth_Next; ?>"
					onclick="JoomliC.changeMonth('<?php echo $modid; ?>', <?php echo htmlspecialchars($params); ?>, '<?php echo $calYear_Next; ?>', '<?php echo $calMonth_Next; ?>')">
				</button>
			</div>

			<div class="ic-calendar-main">
				<div class="ic-calendar-top">
					<?php if ($moduleParams->get('show_week_number', 0)) : ?>
						<div class="ic-calendar-top-week-number"></div>
					<?php endif; ?>
					<?php for ($i = 0; $i < 7; $i++) : ?>
						<?php
						$dayPosition = ($firstDay - $i);

						if ($dayPosition < 0) {
							$dayPosition = $dayPosition + 7;
						}
						?>
						<span class="ic-calendar-top-day"><abbr title="<?php echo $fullDays[$dayPosition]; ?>"><?php echo $shortDays[$dayPosition]; ?></abbr></span>
					<?php endfor; ?>
				</div>

				<div class="ic-calendar-row">
					<?php if ($moduleParams->get('show_week_number', 0)) : ?>
						<div class="ic-calendar-week-number<?php echo $currentWeek; ?>"><?php echo date('W', strtotime($currentDate)); ?></div>
					<?php endif; ?>
					<?php
					for ($d = 1; $d <= $calDisplay; $d++) {
						if (
							($d >= $calMonthFirstDayPosition || $calMonthFirstDayPosition == 1)
							&& $d <= ($calMonthLastDayPosition)
						) {
							// Current date
							$dayCount = ($dayCount < 10) ? '0' . $dayCount : $dayCount;
							$currentDate = $calYear . '-' . $calMonth . '-' . $dayCount;
							$dayDisplay = ltrim($dayCount, '0');

							// Get number of events based on the current date
							$eventNum = 0;
							$dayBgColor = '';

							if (isset($calEvents[$currentDate])) {
								$eventNum = \count($calEvents[$currentDate]);
								$dayBgColor = $calEvents[$currentDate][0]['cat_color'];

								$firstTime = date('H:i', strtotime($calEvents[$currentDate][0]['date']));

								foreach ($calEvents[$currentDate] as $k => $v) {
									$time = date('H:i', strtotime($v['date']));
									if ($time < $firstTime) {
										$dayBgColor = $v['cat_color'];
										$firstTime == $time;
									}
								}
							}

							$textColorClass = self::getTextColorClass($dayBgColor, $dayA11y);

							// Set Event Indicator
							$indicatorClass  = '';

							if ($eventIndicator == 1) {
								$indicatorClass = ' ic-event-1';
							} elseif ($eventIndicator == 2) {
								$numClass = ($eventNum <= 3) ? (($eventNum > 0) ? $eventNum : '0') : '3-plus';
								$indicatorClass = ' ic-event-' . $numClass;
							} elseif ($eventIndicator == 3) {
								$indicatorClass = ' ic-event-number';
							}

							// Set Days
							if (strtotime($currentDate) == strtotime($todayTZ)) {
								if ($eventNum > 0) {
									echo '
										<button class="ic-calendar-day today event' . $indicatorClass . '" onclick="JoomliC.loadEvents(\'' . $modid . '\', \'' . htmlspecialchars($params) . '\', \'' . $currentDate . '\')">
											<span class="ic-calendar-date' . $dateClass . $dayClass . $textColorClass . $textSizeClass . '" style="background-color: ' . $dayBgColor . ';">
												<time datetime="' . $currentDate . '" class="dot-indicator" data-total-events="' . $eventNum . '">' . $dayDisplay . '</time>
											</span>
											<span class="visually-hidden">' . Text::plural('ICAGENDA_EVENT_INDICATOR_STRING', $eventNum) . '</span>
										</button>
									';
								} else {
									echo '
										<div class="ic-calendar-day today">
											<div class="ic-calendar-date' . $dateClass . $dayClass .  $textSizeClass . '">
												<time datetime="' . $currentDate . '">' . $dayDisplay . '</time>
											</div>
										</div>
									';
								}
							} elseif ($eventNum > 0) {
								echo '
									<button class="ic-calendar-day event' . $indicatorClass . '" onclick="JoomliC.loadEvents(\'' . $modid . '\', \'' . htmlspecialchars($params) . '\', \'' . $currentDate . '\');"> 
										<span class="ic-calendar-date' . $dateClass . $dayClass . $textColorClass . $textSizeClass . '" style="background-color: ' . $dayBgColor . ';">
											<time datetime="' . $currentDate . '" class="dot-indicator" data-total-events="' . $eventNum . '">' . $dayDisplay . '</time>
										</span>
										<span class="visually-hidden">' . Text::plural('ICAGENDA_EVENT_INDICATOR_STRING', $eventNum) . '</span>
									</button>
								';
							} else {
								echo '
									<div class="ic-calendar-day no-event"> 
										<span class="ic-calendar-date' . $dateClass .  $textSizeClass . '">
											<time datetime="' . $currentDate . '">' . $dayDisplay . '</time>
										</span>
									</div>
								';
							}
							$dayCount++;
						} elseif (!$thisMonthOnly) {
							if ($d < $calMonthFirstDayPosition) {
								$inactiveCalendarDay = ((($calMonthNbOfDays_Prev - $calMonthFirstDayPosition) + 1) + $d);
								$dayPrev             = ($inactiveCalendarDay < 10) ? '0' . $inactiveCalendarDay : $inactiveCalendarDay;
								$inactiveDate        = $calYear_Prev . '-' . $calMonth_Prev . '-' . $dayPrev;
								$inactiveLabel       = '';
							} else {
								$inactiveCalendarDay = ($d - $calMonthLastDayPosition); 
								$dayNext             = ($inactiveCalendarDay < 10) ? '0' . $inactiveCalendarDay : $inactiveCalendarDay;
								$inactiveDate        = $calYear_Next . '-' . $calMonth_Next . '-' . $dayNext;
								$inactiveLabel       = '';
							}
							// Get number of events based on the current date
							$eventNum = 0;

							if (isset($calEvents[$inactiveDate])) {
								$eventNum = \count($calEvents[$inactiveDate]);
							}

							// Set Event Indicator
							$indicatorClass  = '';

							if ($eventIndicator == 1) {
								$indicatorClass = ' ic-event-1';
							} elseif ($eventIndicator == 2) {
								$numClass = ($eventNum <= 3) ? (($eventNum > 0) ? $eventNum : '0') : '3-plus';
								$indicatorClass = ' ic-event-' . $numClass;
							} elseif ($eventIndicator == 3) {
								$indicatorClass = ' ic-event-number';
							}

							if ($eventNum > 0) {
								echo '
									<button class="ic-calendar-day inactive' . $indicatorClass . '" onclick="JoomliC.loadEvents(\'' . $modid . '\', \'' . htmlspecialchars($params) . '\', \'' . $inactiveDate . '\');">
										<span class="ic-calendar-date' . $dateClass . $dayClass . $textSizeClass . '">
											<time datetime="' . $inactiveDate . '" class="dot-indicator" data-total-events="' . $eventNum . '">' . $inactiveCalendarDay . '</time>
										</span>
										<span class="visually-hidden">' . Text::plural('ICAGENDA_EVENT_INDICATOR_STRING', $eventNum) . '</span>
									</button>
								';
							} else {
								echo '
									<div class="ic-calendar-day inactive">
										<span class="ic-calendar-date' . $dateClass . $dayClass . $textSizeClass . '">
											<time datetime="' . $inactiveDate . '">' . $inactiveCalendarDay . '</time>
										</span>
									</div>
								';
							}
						} else {
							echo '
								<div class="ic-calendar-day inactive">
									<span class="ic-calendar-date' . $dateClass . $dayClass . $textSizeClass . '"></span>
								</div>
							';
						}

						$currentWeek = '';

						if (date('W', strtotime($currentDate . ' + 1 Day')) == date('W')) {
							$currentWeek = ' current-week';
						}

						if ($d%7 == 0 && $d != $calDisplay) {
							echo '</div><div class="ic-calendar-row">';
							if ($moduleParams->get('show_week_number', 0)) {
								echo '<div class="ic-calendar-week-number' . $currentWeek . '">' . date('W', strtotime($currentDate . ' + 1 Day')) . '</div>';
							}
						}

					}
					?>
				</div>
			</div>
		</div>
		<?php

//		$endtime_cal = iCLibrary::getMicrotime();
//		echo '<center style="font-size:8px;">Time to generate the calendar: ' . round($endtime_cal - $starttime_cal, 3) . ' seconds</center>';
	}

	/*
	 * Generate months options list for select box
	 */
	public static function getMonthList($selected = '')
	{
		$options = '';

		$months = [
			'01' => Text::_('JANUARY'),
			'02' => Text::_('FEBRUARY'),
			'03' => Text::_('MARCH'),
			'04' => Text::_('APRIL'),
			'05' => Text::_('MAY'),
			'06' => Text::_('JUNE'),
			'07' => Text::_('JULY'),
			'08' => Text::_('AUGUST'),
			'09' => Text::_('SEPTEMBER'),
			'10' => Text::_('OCTOBER'),
			'11' => Text::_('NOVEMBER'),
			'12' => Text::_('DECEMBER'),
		];

		foreach ($months as $value => $title) {
			$selectedOpt = ($value == $selected) ? ' selected' : '';
			$options.= '<option value="' . $value . '"' . $selectedOpt . '>' . $title . '</option>';
		}

		return $options;
	}

	/*
	 * Generate events list in HTML format
	 */
	public static function getEvents($modid, string $params, $date = '')
	{
		// Load component language files
		$lang = Factory::getLanguage();
		$lang->load('com_icagenda', JPATH_BASE)
			|| $lang->load('com_icagenda', JPATH_ADMINISTRATOR . '/components/com_icagenda');

		// Module Params
		$modid = $modid ?: $this->module->id;
		$module = ModuleHelper::getModuleById((string) $modid);

		$params = $params ?: $module->params;

		$moduleParams = new Registry;
		$moduleParams->loadString($params);

		$date = $date ? date('Y-m-d', strtotime($date)) : date('Y-m-d');

		// Load module language.
		Factory::getApplication()->getLanguage()->load('mod_icagenda_calendar', JPATH_SITE);

		$dayEvents = self::getEventsByDate($modid, $params, $date);

		$dateDay = self::formatDate($date, $moduleParams->get('date_format', ''), $moduleParams->get('date_separator', ''));

		// Calendar dialog box.
		$icDialogHTML = '';

		$icDialogHTML.= '<div id="ic-dialog-header" class="ic-dialog-heading">';
		$icDialogHTML.= '<div class="ic-dialog-heading-date">';
		$icDialogHTML.= '<time datetime="' . date("Y-m-d", strtotime($date)) . '">';
		$icDialogHTML.= '<span class="ic-dialog-date-day">' . $dateDay . '</span>';
		$icDialogHTML.= '</time>';
		$icDialogHTML.= '</div>';
		$icDialogHTML.= '<div class="ic-dialog-heading-close">';
		$icDialogHTML.= '<button id="dialog-close" class="btn btn-secondary btn-sm"><span class="icon-cancel" aria-hidden="true"></span> ' . Text::_('JCLOSE') . '</button>';
		$icDialogHTML.= '</div>';
		$icDialogHTML.= '</div>';

		$icDialogHTML.= '<div id="ic-dialog-body" class="ic-dialog-body">';
		$icDialogHTML.= '<div class="ic-dialog-total-events">' . Text::plural('MOD_ICAGENDA_CALENDAR_DATE_N_EVENTS', \count($dayEvents)) . '</div>';

		// Fetch events based on the specific date 
		if (\count($dayEvents) > 0) {
			$icDialogHTML.= '<ul id="ic-dialog-list" class="ic-dialog-list">';
			$i = 0;
			foreach ($dayEvents as $event) {
				$i++;
				$icDialogHTML.= '<li class="ic-dialog-list-item">';

				// NEW
				$icDialogHTML.= '<div class="ic-dialog-event">';

				$icDialogHTML.= '<div class="ic-dialog-event-image" style="width: 48px">';

				if ($event['image']) {
					$icDialogHTML.= '<img class="ic-dialog-img" src="' . $event['image'] . '" alt="" width="48" height="48" />';
				} else {
					$iconDefault = $event['display_time'] ? 'clock' : 'calendar';
					$icDialogHTML.= '<div class="ic-dialog-img" style="width: 48px; height: 48px; background-color: ' . $event['cat_color'] . ';"><span class="iCicon iCicon-' . $iconDefault . '" style="font-size: 1.5rem; margin-top: calc(50% - .75rem); opacity: .25;"></span></div>';
				}

//				if ($event['display_time']) {
//					$icDialogHTML.= '
//						<div class="ic-dialog-time" style="width: 48px; text-align: center;">
//							' . $event['time'] . '
//						</div>
//					';
//				}

				$icDialogHTML.= '</div>';

				$icDialogHTML.= '<div class="ic-dialog-info" style="border-color: ' . $event['cat_color'] . '">';

				$icDialogHTML.= '<div class="ic-title"><a class="stretched-link" href="' . $event['url'] . '">' . $event['title'] . '</a></div>';

				if ($event['cancelled']) {
					$icDialogHTML.= '<div class="ic-dialog-alert">';
					$icDialogHTML.= '<span class="icon-warning ic-dialog-icon" aria-hidden="true"></span>';
					$icDialogHTML.= '<div class="ic-dialog-cancelled">' . $event['cancelled'] . '</div>';
					$icDialogHTML.= '</div>';
				}

				// Display Time for each date
				if ($event['display_time']) {
					if ($event['full_date']['start_date'] != $event['full_date']['end_date'] && $event['full_date']['end_date']) {
						$icDialogHTML.= '<div class="ic-dialog-time">';
						$icDialogHTML.= '<span class="iCicon-calendar-2 ic-dialog-icon" aria-hidden="true" aria-labelledby="ic-time-label">';
						$icDialogHTML.= '<span id="ic-time-label" hidden>' . Text::_('MOD_ICAGENDA_CALENDAR_TIME_LABEL') . '</span>';
						$icDialogHTML.= '</span> ';
						$icDialogHTML.= $event['full_date']['start_date'];
						if ($event['display_time'] && $event['full_date']['start_time']) {
							$icDialogHTML.= ' ' . $event['full_date']['start_time'];
						}
						if ($event['full_date']['end_date']) {
							$icDialogHTML.= ' - ' . $event['full_date']['end_date'];
							if ($event['display_time'] && $event['full_date']['end_time']) {
								$icDialogHTML.= ' ' . $event['full_date']['end_time'];
							}
						}
						$icDialogHTML.= '</div>';
					} else {
						$icDialogHTML.= '<div class="ic-dialog-time">';
						$icDialogHTML.= '<span class="iCicon-clock ic-dialog-icon" aria-hidden="true" aria-labelledby="ic-time-label">';
						$icDialogHTML.= '<span id="ic-time-label" hidden>' . Text::_('MOD_ICAGENDA_CALENDAR_TIME_LABEL') . '</span>';
						$icDialogHTML.= '</span> ';
						$icDialogHTML.= '<span> ';
						$icDialogHTML.= $event['full_date']['start_time'];
						if ($event['full_date']['end_time']) {
							$icDialogHTML.= ' - ' . $event['full_date']['end_time'];
						}
						$icDialogHTML.= '</span> ';
						$icDialogHTML.= '</div>';
					}
				}

				// Display Category title
				if ($event['cat_title']) {
					$icDialogHTML.= '<div class="ic-dialog-category">';
					$icDialogHTML.= '<span class="iCicon-folder ic-dialog-icon" aria-hidden="true" aria-labelledby="ic-category-label">';
					$icDialogHTML.= '<span id="ic-category-label" hidden>' . Text::_('MOD_ICAGENDA_CALENDAR_CATEGORY_LABEL') . '</span>';
					$icDialogHTML.= '</span> ';
					$icDialogHTML.= '<span> ';
					$icDialogHTML.= $event['cat_title'];
					$icDialogHTML.= '</span>';
					$icDialogHTML.= '</div>';
				}

				// Display Venue Name, City and/or Country for each date
				if ($event['venue'] || $event['city'] || $event['country']) {
					$icDialogHTML.= '<div class="ic-dialog-venue">';

					$icDialogHTML.= '<span class="iCicon-location ic-dialog-icon" aria-hidden="true" aria-labelledby="ic-venue-label">';
					$icDialogHTML.= '<span id="ic-venue-label" hidden>' . Text::_('MOD_ICAGENDA_CALENDAR_VENUE_LABEL') . '</span>';
					$icDialogHTML.= '</span> ';

					$icDialogHTML.= '<div class="ic-dialog-venue-info">';

					// Display Venue Name
					if ($event['venue']) {
						$icDialogHTML.= '<div class="ic-dialog-venue-name">';
						$icDialogHTML.= $event['venue'];
						$icDialogHTML.= '</div>';
					}

					// Display City
					if ($event['city']) {
						$icDialogHTML.= $event['city'];
					}

					// Display Country
					if ($event['country'] && $event['city']) {
						$icDialogHTML.= ', ' . $event['country'];
					} elseif ($event['country'] && !$event['city']) {
						$icDialogHTML.= $event['country'];
					}

					$icDialogHTML.= '</div>';

					$icDialogHTML.= '</div>';
				}

				// Display Feature Tags
				if (!empty($event['feature_tags'])) {
					$icDialogHTML.= '<div class="ic-dialog-feature-tags-container">';

					$tagClass = ($event['show_feature_tags'] != 1) ? ' ic-badge' : '';

					foreach ($event['feature_tags'] as $ft) {
						$iconAlt = $ft['icon_alt'] ? $ft['icon_alt'] : $ft['title'];

						$icDialogHTML.= '<div class="ic-dialog-feature-tag' . $tagClass . '">';

						if ($event['show_feature_tags'] == 1) {
							$icDialogHTML.= '<img src="' . $event['feature_icons_root'] . $ft['icon'] . '" alt="' . $iconAlt . '">';
						} elseif ($event['show_feature_tags'] == 2) {
							$icDialogHTML.= '<img src="' . $event['feature_icons_root'] . $ft['icon'] . '" alt="" />';
							$icDialogHTML.= '<div class="ic-dialog-feature-tag-title ic-ps-2">' . $ft['title'] . '</div>';
						} elseif ($event['show_feature_tags'] == 3) {
							$icDialogHTML.= '<div class="ic-dialog-feature-tag-title">' . $ft['title'] . '</div>';
						}

						$icDialogHTML.= '</div>';
					}

					$icDialogHTML.= '</div>';
				}

				// Display Intro Text (Short Description)
				if ($event['introtext']) {
					$icDialogHTML.= '
						<div class="ic-dialog-desc">
							' . $event['introtext'] . '
						</div>
					';
				}

				// Display Registration Information
				if ($event['registration_info'] || $event['statusMessage']) {
					$icDialogHTML.= '<div class="ic-dialog-registration">';

					if ($event['registration_info']) {
						$icDialogHTML.= '<div class="ic-dialog-tickets">';

						if ($event['ticketsRegistered'] && $event['ticketsTotal']) {
							$icDialogHTML.= '<span class="iCicon-users ic-dialog-icon" aria-hidden="true"></span> <span>'
								. Text::sprintf('MOD_ICAGENDA_CALENDAR_PEOPLE_REGISTERED_OUT_OF', $event['ticketsRegistered'], $event['ticketsTotal'])
								. ' ' . Text::plural('MOD_ICAGENDA_CALENDAR_TICKETS_LEFT', $event['ticketsAvailable'])
								. '</span>';
						} elseif ($event['ticketsRegistered']) {
							$icDialogHTML.= '<span class="iCicon-users ic-dialog-icon" aria-hidden="true"></span> <span>'
								. Text::plural('MOD_ICAGENDA_CALENDAR_PEOPLE_REGISTERED', $event['ticketsRegistered'])
								. '</span>';
						} elseif ($event['ticketsTotal']) {
							$icDialogHTML.= '<span class="iCicon-users ic-dialog-icon" aria-hidden="true"></span> <span>'
								. Text::plural('MOD_ICAGENDA_CALENDAR_TICKETS_AVAILABLE', $event['ticketsAvailable'])
								. '</span>';
						} else {
							$icDialogHTML.= '<span class="iCicon-users ic-dialog-icon" aria-hidden="true"></span> <span class="text-muted">'
								. Text::plural('MOD_ICAGENDA_CALENDAR_PEOPLE_REGISTERED', $event['ticketsRegistered'])
								. '</span>';
						}
						$icDialogHTML.= '</div>';
					} elseif ($event['statusMessage']) {
						$icDialogHTML.= '<div class="ic-dialog-registration-status">'
							. '<span class="iCicon-register-2 ic-dialog-icon" aria-hidden="true"></span> '
							. '<span class="ic-dialog-registration-closed">' . $event['statusMessage'] . '</span>'
							. '</div>';
					}

					$icDialogHTML.= '</div>';
				}

				$icDialogHTML.= '</div>';

				$icDialogHTML.= '</div>';

				if ($event['regButton'] && !$event['statusMessage']) {
					$icDialogHTML.= '<div class="ic-dialog-actions-container">';
					$icDialogHTML.= '<div class="ic-dialog-event-image" style="width: 48px">';
					$icDialogHTML.= '</div>';
					$icDialogHTML.= '<div class="ic-dialog-actions" style="border-color: ' . $event['cat_color'] . '">';
					$icDialogHTML.= $event['regButton'];
					$icDialogHTML.= '</div>';
					$icDialogHTML.= '</div>';
				}
			}

			$icDialogHTML.= '</ul>';
//			$icDialogHTML.= '<div class="ic-dialog-footer">';
//			$icDialogHTML.= '<span style="opacity: .75;">' . Text::_('MOD_ICAGENDA_CALENDAR_END_OF_EVENTS_LIST_NOTE') . '</span>';
//			$icDialogHTML.= '</div>';

			$icDialogHTML.= '</div>';
		}

		echo $icDialogHTML;
	}

	/*
	 * Get events by date
	 */
	public static function getEventsCurrentCal($modid, string $params, $dateStart = '', $dateEnd = '')
	{
		if (!$dateStart && !$dateEnd) {
			return false;
		}

		// Component Params
		$iCparams = ComponentHelper::getParams('com_icagenda');

		// Module Params
		$modid = $modid ?: $this->module->id;
		$module = ModuleHelper::getModuleById((string) $modid);

		$params = $params ?: $module->params;

		$moduleParams = new Registry;
		$moduleParams->loadString($params);


		// Get module options
		$events_menu_item         = $moduleParams->get('events_menu_item', '');
		$events_menu_item_filters = $moduleParams->get('events_menu_item_filters', 0);
		$category_filter          = $moduleParams->get('category_filter', '');
		$category_selection       = $moduleParams->get('category_selection', '');

		// Global Joomla API objects
		$app  = Factory::getApplication();
		$lang = Factory::getLanguage();
		$menu = $app->getMenu();

		// Itemid Request (automatic detection of the first iCagenda menu-link, by menuID)
		$iC_list_menus = icagendaMenus::iClistMenuItemsInfo();

		// Set variables
		$eventTimeZone  = null;
		$datetime_today = HTMLHelper::date('now', 'Y-m-d H:i');

		// Get the events
		$db    = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->createQuery()
			->select('e.id,
				e.title,
				e.startdate,
				e.enddate,
				e.weekdays,
				e.next,
				e.dates,
				e.catid,
				c.color as cat_color
			')
			->from($db->qn('#__icagenda_events').' AS e')
			->leftJoin($db->qn('#__icagenda_category').' AS c ON ' . $db->qn('c.id') . ' = ' . $db->qn('e.catid'));

		// Where Category is Published
		$query->where('c.state = 1');

		// Where State is Published
		$query->where('e.state = 1');

		// Where event is Approved
		$query->where('e.approval = 0');

		// Filter next date
		$query->where(
			'('
				. '('
					. $db->qn('e.next') . ' >= ' . $db->q($dateStart)
					. ' AND ' . $db->qn('e.next') . ' < ' . $db->q($dateEnd)
				. ')'
				. ' OR ('
					. $db->qn('e.startdate') . ' <= ' . $db->q($dateEnd)
					. ' AND ' . $db->qn('e.enddate') . ' >= ' . $db->q($dateStart)
					. ' AND ' . $db->qn('e.weekdays') . ' = ""'
				. ')'
				. ' OR ('
					. $db->qn('e.startdate') . ' <= ' . $db->q($dateEnd)
					. ' AND ' . $db->qn('e.enddate') . ' >= ' . $db->q($dateStart)
					. ' AND ' . $db->qn('e.weekdays') . ' <> ""'
				. ')'
				. ' OR ('
					. $db->qn('e.dates') . ' <> ""'
//					. ' AND ' . $db->qn('e.next') . ' >= ' . $db->q($dateStart)
//					. ' AND ' . $db->qn('e.next') . ' < ' . $db->q($dateEnd)
					// Will be improved for single dates in iCagenda 5 with the new dates system.
				. ')'
			. ')'
		);

		// Filter by categories to be displayed
		if ($category_filter != '') {
			$catFilter = ! is_array($category_selection) ? [$category_selection] : $category_selection;

			// Note: zero value kept for Joomla 2.5 B/C (option All categories not used on J3 sites)
			if ( $catFilter && !\in_array('', $catFilter)) {
				$cats_option = implode(', ', $catFilter);

				if ($category_filter == 1) {
					$query->where('e.catid IN (' . $cats_option . ')');
				} else {
					$query->where('e.catid NOT IN (' . $cats_option . ')');
				}
			}
		}

		// Check Access Levels
		$user       = Factory::getApplication()->getIdentity();
		$userID     = $user->id;
		$userLevels = $user->getAuthorisedViewLevels();
		$userGroups = $user->getAuthorisedGroups();

		$userAccess = implode(', ', $userLevels);

		if (!\in_array('8', $userGroups)) {
			$query->where('e.access IN (' . $userAccess . ')');
		}

		// Filter by language
		$query->where('e.language IN (' . $db->q(Factory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');

		// Run the query
		$db->setQuery($query);

		// Invoke the query
		$result = $db->loadObjectList();

		// Get events for the current calendar view.
		$eventsCurrentCal = [];

		foreach ($result as $item) {
			// Frontend Module Check Next.
			$item->next = icagendaEvent::getNextDate($item);

			// List event dates.
			$AllDates = [];

			// Get list of valid single dates for this event.
			$allSingleDates_array = self::getDatelist($item->dates);

			sort($allSingleDates_array);

			foreach ($allSingleDates_array as $sd) {
				$this_date = HTMLHelper::date($sd, 'Y-m-d', null);

				if (strtotime($this_date) >= strtotime($dateStart)
					&& strtotime($this_date) < strtotime($dateEnd)
				) {
					$AllDates[] = $sd;
				}
			}

			// Get WeekDays Array
			$WeeksDays = iCDatePeriod::weekdaysToArray($item->weekdays);

			// Get Period Dates
			$startDate_TZ = iCDate::isDate($item->startdate)
				? HTMLHelper::date($item->startdate, 'Y-m-d H:i', $eventTimeZone)
				: false;
			$endDate_TZ   = iCDate::isDate($item->enddate)
				? HTMLHelper::date($item->enddate, 'Y-m-d H:i', $eventTimeZone)
				: false;

			$periodDates  = iCDatePeriod::listDates($item->startdate, $item->enddate); // UTC

			$fullDateRange = $moduleParams->get('full_date_range_event', '');

			// Check the period if it is a full period with no days of the week selected.
			$fullPeriod = ($item->weekdays || $item->weekdays == '0') ? false : true;

			if ( ! empty($periodDates)) {
				if ($fullDateRange == 'onlystart' && $fullPeriod) {
					if (strtotime($startDate_TZ) >= strtotime($dateStart)
						&& strtotime($startDate_TZ) < strtotime($dateEnd)
					) {
						$AllDates[] = date('Y-m-d H:i', strtotime($item->startdate));
					}
				} else {
					foreach ($periodDates as &$Dat) {
						$this_date = HTMLHelper::date($Dat, 'Y-m-d', null);

						if (\in_array(date('w', strtotime($Dat)), $WeeksDays)) {
							$SingleDate = date('Y-m-d H:i', strtotime($Dat));

							if (strtotime($this_date) >= strtotime($dateStart)
								&& strtotime($this_date) < strtotime($dateEnd)
							) {
								$AllDates[] = $SingleDate;
							}
						}
					}
				}
			}

			rsort($AllDates);

			$event = array(
				'id'        => (int)$item->id,
				'title'     => $item->title,
				'cat_color' => $item->cat_color,
			);

			// Get List of Dates
			if (is_array($event)) {
				$past_dates = 0;

				foreach ($AllDates as $d) {
					// Control if date is past
					if (strtotime($d) < strtotime($datetime_today)) {
						$past_dates = $past_dates + 1;
					}
				}

				unset($d);

				foreach ($AllDates as $d) {
					$event_filters = [
						'date'     => $d,
					];

					// If use menu item filters
					if ($events_menu_item_filters === '1') {
						$linkid = $events_menu_item
								? icagendaMenus::displayEventItemid($events_menu_item, $event_filters)
								: icagendaMenus::thisEventItemid($d, $item->catid, $iC_list_menus);
					} else {
						$linkid = $events_menu_item ? $events_menu_item : icagendaMenus::thisEventItemid($d, $item->catid, $iC_list_menus);
						$linkid = $linkid ? $linkid : $menu->getDefault($lang->getTag())->id;
					}

					$event = array_merge($event, $event_filters);

					if ($linkid) {
						$dateCal = date('Y-m-d', strtotime($event['date']));
						$eventsCurrentCal[$dateCal][] = $event;
					}
				}

				unset($d);
			}
		}

		ksort($eventsCurrentCal);

		return $eventsCurrentCal;
	}

	/*
	 * Get events by date
	 */
	public static function getEventsByDate($modid, string $params, $date = '')
	{
		// Global Joomla API objects
		$app  = Factory::getApplication();
		$lang = Factory::getLanguage();
		$menu = $app->getMenu();

		// Component Params
		$iCparams = ComponentHelper::getParams('com_icagenda');

		// Module Params
		$modid = $modid ?: $this->module->id;
		$module = ModuleHelper::getModuleById((string) $modid);

		$params = $params ?: $module->params;

		$moduleParams = new Registry;
		$moduleParams->loadString($params);

		// Get module options
		$events_menu_item         = $moduleParams->get('events_menu_item', '');
		$events_menu_item_filters = $moduleParams->get('events_menu_item_filters', 0);
		$show_time                = $moduleParams->get('show_time', 1);
		$intro_text               = $moduleParams->get('intro_text', 0);
		$category_filter          = $moduleParams->get('category_filter', '');
		$category_selection       = $moduleParams->get('category_selection', '');
		$show_feature_tags        = $moduleParams->get('show_feature_tags', '');
		$feature_tags_icon_size   = $moduleParams->get('feature_tags_icon_size', '');
		$registration_tickets     = $moduleParams->get('registration_tickets', '');
		$registration_status      = $moduleParams->get('registration_status', '');

		$showRegTickets = \is_array($registration_tickets) ? $registration_tickets : explode(',', $registration_tickets);
		$showRegStatus  = \is_array($registration_status) ? $registration_status : explode(',', $registration_status);

		// Set variables
		$eventTimeZone  = null;
		$datetime_today = HTMLHelper::date('now', 'Y-m-d H:i');

		// Itemid Request (automatic detection of the first iCagenda menu-link, by menuID)
		$iC_list_menus = icagendaMenus::iClistMenuItemsInfo();

		// Check if GD is enabled on the server
		if (extension_loaded('gd') && function_exists('gd_info')) {
			$thumb_generator = $iCparams->get('thumb_generator', 1);
		} else {
			$thumb_generator = 0;
		}

		// Get the events
		$db    = Factory::getContainer()->get('DatabaseDriver');
		$query = $db->createQuery();

		// Build the query
		$query->select('e.*,
				e.place as venue_name,
				c.title as cat_title,
				c.alias as cat_alias,
				c.color as cat_color,
				c.ordering as cat_order
			')
			->from($db->qn('#__icagenda_events') . ' AS e')
			->leftJoin($db->qn('#__icagenda_category').' AS c ON ' . $db->qn('c.id') . ' = ' . $db->qn('e.catid'));

		// Where Category is Published
		$query->where($db->qn('c.state') . ' = 1');

		// Where State is Published
		$query->where($db->qn('e.state') . ' = 1');

		// Where event is Approved
		$query->where($db->qn('e.approval') . ' = 0');

		// Filter next date
		if ($date) {
			$dayStart = date('Y-m-d', strtotime($date));
			$dayEnd   = date('Y-m-d', strtotime($date . ' + 1 DAY'));
			$dateControlMin = date('Y-m', strtotime($date . ' - 1 MONTH')) . '-01';
			$dateControlMax = date('Y-m', strtotime($dateControlMin . ' + 2 MONTHS')) . '-01';
			$dateWeekDay = date('w', strtotime($date));

			$query->where(
				'('
					. '('
						. $db->qn('e.dates') . ' <> ""'
						. ' AND ' . $db->qn('e.dates') . ' LIKE ' . $db->q('%' . $dayStart . '%')
					. ')'
					. ' OR ('
						. $db->qn('e.startdate') . ' <= ' . $db->q($dayEnd)
						. ' AND ' . $db->qn('e.enddate') . ' >= ' . $db->q($dayStart)
						. ' AND ' . $db->qn('e.weekdays') . ' = ""'
					. ')'
					. ' OR ('
						. $db->qn('e.startdate') . ' <= ' . $db->q($dayEnd)
						. ' AND ' . $db->qn('e.enddate') . ' >= ' . $db->q($dayStart)
						. ' AND ' . $db->qn('e.weekdays') . ' <> ""'
					. ')'
				. ')'
			);
		}

		// Filter by categories to be displayed
		if ($category_filter != '') {
			$catFilter = ! is_array($category_selection) ? [$category_selection] : $category_selection;

			if ( $catFilter && !\in_array('', $catFilter)) {
				$cats_option = implode(', ', $catFilter);

				if ($category_filter == 1) {
					$query->where('e.catid IN (' . $cats_option . ')');
				} else {
					$query->where('e.catid NOT IN (' . $cats_option . ')');
				}
			}
		}

		// Check Access Levels
		$user		= Factory::getApplication()->getIdentity();
		$userID		= $user->id;
		$userLevels	= $user->getAuthorisedViewLevels();
		$userGroups = $user->getAuthorisedGroups();

		$userAccess = implode(', ', $userLevels);

		if ( ! \in_array('8', $userGroups)) {
			$query->where('e.access IN (' . $userAccess . ')');
		}

		// Filter by language
		$query->where('e.language IN (' . $db->q(Factory::getLanguage()->getTag()) . ',' . $db->q('*') . ')');

		// Features - extract the number of displayable icons per event
		$query->select('feat.count AS features');
		$sub_query = $db->createQuery();
		$sub_query->select('fx.event_id, COUNT(*) AS count');
		$sub_query->from('`#__icagenda_feature_xref` AS fx');
		$sub_query->innerJoin("`#__icagenda_feature` AS f ON fx.feature_id=f.id AND f.state=1 AND f.icon<>'-1'");
		$sub_query->group('fx.event_id');
		$query->leftJoin('(' . (string) $sub_query . ') AS feat ON e.id=feat.event_id');

		// Registrations total
		$query->select('r.count AS reg_people, r.date AS reg_date');
		$sub_query = $db->createQuery();
		$sub_query->select('r.eventid, sum(r.people) AS count, r.date AS date');
		$sub_query->from('`#__icagenda_registration` AS r');
		$sub_query->where('r.state > 0');
		$sub_query->group('r.eventid');
		$query->leftJoin('(' . (string) $sub_query . ') AS r ON e.id=r.eventid');

		// Run the query
		$db->setQuery($query);

		// Invoke the query
		$result = $db->loadObjectList();

		foreach ($result as &$record) {
			$record_registered = [];
			$registrations     = icagendaEventsData::registeredList(); // @todo: migrate this function.

			foreach ($registrations as $reg_by_event) {
				$ex_reg_by_event = explode('@@', $reg_by_event);

				if ($ex_reg_by_event[0] == $record->id) {
					$record_registered[] = $ex_reg_by_event[0] . '@@' . $ex_reg_by_event[1] . '@@' . $ex_reg_by_event[2] . '@@' . $ex_reg_by_event[3];
				}
			}

			$record->registered = $record_registered;
		}

		$eventsOfDay = [];
		$i = 0;

		foreach ($result as $item) {
			// Frontend Module Check Next
			$item->next = icagendaEvent::getNextDate($item);

			// Extract the feature details, if needed.
			$feature_tags = [];

			if ($show_feature_tags) {
				if (is_null($item->features)) {
					$item->features = [];
				} else {
					$item->features = icagendaEvents::featureIcons($item->id);
				}

				if (isset($item->features) && \is_array($item->features)) {
					foreach ($item->features as &$feature) {
						// Note: control if title is set, b/c 3.9. To be removed in 4.0-stable.
						$feature_tags[] = [
							'title'     => (isset($feature->title) ? $feature->title : $feature->icon_alt),
							'icon'      => $feature->icon,
							'icon_alt'  => $feature->icon_alt,
							'icon_size' => $feature_tags_icon_size,
						];
					}
				}
			}

			// list calendar dates
			$AllDates = [];

			// Get list of valid single dates for this event
			$allSingleDates_array = self::getDatelist($item->dates);

			sort($allSingleDates_array);

			// If Single Dates, added to all dates for this event
//			if (isset($datemultiplelist)
//				&& $datemultiplelist != NULL
//				&& is_array($datemultiplelist))
//			{
//				$allSingleDates_array = array_merge($AllDates, $datemultiplelist);
//			}

			foreach ($allSingleDates_array as $sd) {
				$this_date = HTMLHelper::date($sd, 'Y-m-d', null);

				if (strtotime($this_date) >= strtotime($dayStart)
					&& strtotime($this_date) < strtotime($dayEnd)
				) {
					$AllDates[] = $sd;
				}
			}

			// Get WeekDays Array
			$WeeksDays = iCDatePeriod::weekdaysToArray($item->weekdays);

			// Get Period Dates
			$startDate_TZ = iCDate::isDate($item->startdate)
				? HTMLHelper::date($item->startdate, 'Y-m-d H:i', $eventTimeZone)
				: false;
			$endDate_TZ   = iCDate::isDate($item->enddate)
				? HTMLHelper::date($item->enddate, 'Y-m-d H:i', $eventTimeZone)
				: false;
			$periodDates  = iCDatePeriod::listDates($item->startdate, $item->enddate); // UTC

			$fullDateRange = $moduleParams->get('full_date_range_event', '');

			// Check the period if individual dates
			$fullPeriod = ($item->weekdays || $item->weekdays == '0') ? false : true;

			if ( ! empty($periodDates)) {
				if ($fullDateRange == 'onlystart' && $fullPeriod) {
					if (strtotime($startDate_TZ) >= strtotime($dayStart)
						&& strtotime($startDate_TZ) < strtotime($dayEnd)
					) {
						$AllDates[] = date('Y-m-d H:i', strtotime($item->startdate));
					}
				} else {
					foreach ($periodDates as &$Dat) {
						$this_date = HTMLHelper::date($Dat, 'Y-m-d', null);

						if (\in_array(date('w', strtotime($Dat)), $WeeksDays)) {
							$SingleDate = date('Y-m-d H:i', strtotime($Dat));

							if (strtotime($this_date) >= strtotime($dayStart)
								&& strtotime($this_date) < strtotime($dayEnd)
							) {
								$AllDates[] = $SingleDate;
							}
						}
					}
				}
			}

			rsort($AllDates);

			$filtering_shortDesc = $moduleParams->get('filtering_shortDesc', '');

			$limit = $moduleParams->get('introtext_limit', '')
				? $moduleParams->get('introtext_limit_custom', '')
				: false;

			$descShort = icagendaEvents::shortDescription($item->desc, true, $filtering_shortDesc, $limit);

			/**
			 * Get Thumbnail
			 */
			// Set if run iCthumb
			$img = HTMLHelper::cleanImageURL($item->image);

			$img_url = $img->url;

			if ($img_url
				&& $thumb_generator == 1
			) {
				// Generate small thumb if not exist
				$thumb_img = icagendaThumb::sizeXSmall($img_url);
			} elseif ($img_url
				&& $thumb_generator == 0
			) {
				$thumb_img = $img_url;
			} else {
				$thumb_img = $img_url ? 'media/com_icagenda/images/nophoto.jpg' : '';
			}

			$evtParams = new Registry($item->params);

			// Display Intro Text
			if ($intro_text == '1') {
				// Short Description
				$introText = $item->shortdesc ? $item->shortdesc : false;
			} elseif ($intro_text == '2') {
				// Auto-Introtext
				$introText = $descShort ? $descShort : false;
			} elseif ($intro_text == '0') {
				// Hide
				$introText = false;
			} else {
				// Auto (First Short Description, if does not exist, Auto-generated short description from the full description. And if does not exist, will use meta description if not empty)
				$e_shortdesc = $item->shortdesc ? $item->shortdesc : $descShort;
				$introText   = $e_shortdesc ? $e_shortdesc : $item->metadesc;
			}

			// Display Registration Infos
			$eventRegStatus    = $evtParams->get('statutReg', $iCparams->get('statutReg', '0'));
			$registration_info = ($eventRegStatus == 1) ? $moduleParams->get('show_registration_info', 1) : '';

			$maxReg  = $registration_info ? $evtParams->get('maxReg', '1000000') : false;
			$typeReg = $registration_info ? $evtParams->get('typeReg', '1') : false;

			$reg_deadline = $evtParams->get('reg_deadline', $iCparams->get('reg_deadline', '1'));

			$eventIsCancelled = icagendaEvent::cancelledButton($item->id);

//			$eventTitle = $eventIsCancelled
//				? $item->title . '<div class="ic-event-cancelled text-danger"><span class="icon-warning"></span> ' . $eventIsCancelled . '</div>'
//				: $item->title;
			$imagesPath = icagendaMedia::iCagendaImagesPath();

			$category = $moduleParams->get('show_category', 0) ? $item->cat_title : '';
			$venue    = $moduleParams->get('show_venue_name', 1) ? $item->venue_name : '';
			$city     = $moduleParams->get('show_city', 1) ? $item->city : '';
			$country  = $moduleParams->get('show_country', 1) ? $item->country : '';

			$event = [
				'id'                 => (int) $item->id,
				'title'              => $item->title,
				'cancelled'          => $eventIsCancelled,
				'image'              => $thumb_img,
				'file'               => $item->file, // Not used
				'address'            => $item->address, // Not used

				'venue'              => $venue,
				'city'               => $city,
				'country'            => $country,

				'description'        => $item->desc, // Not used
				'introtext'          => $introText,
				'cat_title'          => $category,
				'cat_order'          => $item->cat_order, // Not used
				'cat_color'          => $item->cat_color,
				'no_image'           => Text::_('MOD_ICCALENDAR_NO_IMAGE'), // Not used
				'params'             => $item->params, // Not used

				'show_feature_tags'  => $show_feature_tags,
				'feature_tags'       => $feature_tags,
				'feature_icons_root' => Uri::base() . "{$imagesPath}/feature_icons/{$feature_tags_icon_size}/",
			];

			// Get Option Dislay Time
			$displayTime = isset($item->displaytime) ? $item->displaytime : '';

			$events_per_day	= [];

			$countEventDates = \count($AllDates);

			// Get List of Dates
			if (\is_array($event)) {
				$past_dates = 0;

				foreach ($AllDates as $d) {
					// Control if date is past
					if (strtotime($d) < strtotime($datetime_today)) {
						$past_dates = $past_dates + 1;
					}
				}

				unset($d);

				$events_menu_item = is_numeric($events_menu_item) ? $events_menu_item : '';

				foreach ($AllDates as $d) {
					$i++;

					$event_filters = [
						'date_i'   => $d . '_' . $i,
						'catid'    => $item->catid,
						'language' => $item->language,
						'access'   => $item->access,
					];

					// If use menu item filters
					if ($events_menu_item_filters === '1') {
						$linkid = $events_menu_item
								? icagendaMenus::displayEventItemid($events_menu_item, $event_filters)
								: icagendaMenus::thisEventItemid($d, $item->catid, $iC_list_menus);
					} else {
						$linkid = $events_menu_item ? $events_menu_item : icagendaMenus::thisEventItemid($d, $item->catid, $iC_list_menus);
						$linkid = $linkid ? $linkid : $menu->getDefault($lang->getTag())->id;
					}

					$eventnumber	= $item->id ? $item->id : null;
					$event_slug		= $item->alias ? $item->id . ':' . $item->alias : $item->id;

					$thisDateUTC  = date('Y-m-d H:i', strtotime($d));

					// Set variable date-alias in url & registration deadline datetime
					if ($fullPeriod && \in_array($thisDateUTC, $periodDates)) {
						$regDate = '';
						$vars    = [];

						$regDeadlineDatetime	= ($reg_deadline == '2')
												? HTMLHelper::date($item->enddate, 'Y-m-d H:i:s', false)
												: HTMLHelper::date($item->startdate, 'Y-m-d H:i:s', false);
					} else {
						$regDate = date('Y-m-d H:i:s', strtotime($d));
						$vars    = ['date' => iCDate::dateToAlias($d, 'Y-m-d-H-i')];

						if ($reg_deadline == '2') {
							$regDeadlineDatetime = (\in_array($thisDateUTC, $periodDates))
								? HTMLHelper::date($d, 'Y-m-d', false) . ' ' . HTMLHelper::date($item->enddate, 'H:i:s', false)
								: HTMLHelper::date($d, 'Y-m-d', false) . ' 23:59:59';
						} else {
							$regDeadlineDatetime = HTMLHelper::date($d, 'Y-m-d H:i:s', false);
						}
					}

					if ($show_time) {
						$event['display_time'] = $displayTime;
					} else {
						$event['display_time'] = '';
					}

					$event['full_date'] = icagendaEvent::getDateArray($d, $item);
					$event['url']       = icagendaEvent::url($item->id, $item->alias, $linkid, $vars);

					$event = array_merge($event, $event_filters);

					$ticketsRegistered  = $registration_info
						? icagendaRegistration::getRegisteredTotal($item->id, $regDate, $typeReg)
						: false;
					$ticketsTotal  = ($registration_info && $maxReg != '1000000')
						? $maxReg
						: false;
					$ticketsAvailable = ($registration_info && $maxReg)
						? ($maxReg - $ticketsRegistered)
						: false;

					$canRegister = (HTMLHelper::date('Now', 'Y-m-d H:i:s', false) <= $regDeadlineDatetime);

					$statusMessage = '';
//					$showStatus    = '';
					$showStatus    = 1;

					if ($evtParams->get('event_cancelled', 0)) {
						$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_CLOSED');

//						if (\in_array('closed', $showRegStatus)) {
//							$showStatus = 1;
//						}

					} elseif (
						$typeReg == '2'
						&& $ticketsAvailable <= 0
					) {
						// Registration for all dates, and no ticket left
						$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_FULL');

//						if (\in_array('full', $showRegStatus)) {
//							$showStatus = 1;
//						}

					} elseif (
						$ticketsAvailable <= 0
					) {
						// Registration by date, and no ticket left
						if ($regDeadlineDatetime < $datetime_today) {
							// Expired deadline, registration closed
							$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_CLOSED');

//							if (\in_array('closed', $showRegStatus)) {
//								$showStatus = 1;
//							}
						} else {
							// Upcoming deadline, but no tickets left for this date
							$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_DATE_NO_TICKETS_LEFT');

//							if (\in_array('full', $showRegStatus)) {
//								$showStatus = 1;
//							}
						}

					} elseif (
						$typeReg == '2'
						&& ($reg_deadline != '2'
							&& ((iCDate::isDate($startDate_TZ) && $startDate_TZ < $datetime_today)
							|| (isset($allSingleDates_array[0]) && $allSingleDates_array[0] < $datetime_today))
						)
					) {
						// Registration for all dates + Registration until START date + first date in the past.
						$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_CLOSED');

//						if (\in_array('closed', $showRegStatus)) {
//							$showStatus = 1;
//						}

					} elseif (
						$typeReg == '2'
						&& ($reg_deadline == '2'
							&& ((iCDate::isDate($endDate_TZ) && $endDate_TZ < $datetime_today)
							|| (iCDate::isDate(end($allSingleDates_array)) && end($allSingleDates_array) < $datetime_today))
						)
					) {
						// Registration for all dates + Registration until END date + first date in the past.
						$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_CLOSED');

//						if (\in_array('closed', $showRegStatus)) {
//							$showStatus = 1;
//						}

					} elseif (
						! $canRegister
						&& $typeReg != '2'
					) {
						// Registration by date, and registration deadline is over
						if ($ticketsAvailable <= 0 && $countEventDates > 1
							&& $past_dates < $countEventDates
						) {
							$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_DATE_NO_TICKETS_LEFT');

//							if (\in_array('full', $showRegStatus)) {
//								$showStatus = 1;
//							}
						} else {
							$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_CLOSED');

//							if (\in_array('closed', $showRegStatus)) {
//								$showStatus = 1;
//							}
						}

					} elseif (
						$ticketsTotal
						&& $canRegister
						&& $typeReg != '2'
					) {
						// @todo : test last change > can or cannot register ? (check if needed)
						if ($ticketsAvailable <= 0) {
							$statusMessage = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_DATE_NO_TICKETS_LEFT');
						}

//						if (\in_array('full', $showRegStatus)) {
//							$showStatus = 1;
//						}

					}

					if (!$registration_info) {
						$statusMessage = '';
					}
					if ($statusMessage) {
						$registration_info = false;
					}

//					if (
//						!$regStatus
//						&& \in_array('open', $showRegStatus)
//						&& $canRegister
//					) {
//						$regStatus = Text::_('MOD_ICAGENDA_CALENDAR_REGISTRATION_OPEN');
//					}

					// Set registration button
					$registrationButton = '';

					if ($moduleParams->get('show_registration_button', 0)
						&& !$evtParams->get('event_cancelled', '')
					) {
						// Convert event parameter fields to objects.
						$registry = new Registry;
						$registry->loadString($evtParams);

						// Merge Event params to component params
						$item->params = clone $iCparams;
						$item->params->merge($registry);

						$evt = date('Y-m-d H:i:s', strtotime($d));

						$registrationButton = icagendaRegistration::registerButton($item, $evt, $linkid);
					}

					$reg_infos = [
						'registration_info' => $registration_info,
//						'ticketsRegistered' => (\in_array('registered', $showRegTickets) ? $ticketsRegistered : false),
//						'ticketsTotal'      => (\in_array('total', $showRegTickets) ? $ticketsTotal : false),
//						'ticketsAvailable'  => (\in_array('available', $showRegTickets) ? $ticketsAvailable : false),
						'ticketsRegistered' => $ticketsRegistered,
						'ticketsTotal'      => $ticketsTotal,
						'ticketsAvailable'  => $ticketsAvailable,
//						'regStatus'         => $regStatus,
//						'regDate'           => $evt,
						'regButton'         => $registrationButton,
						'statusMessage'     => $statusMessage,
						'showStatus'        => $showStatus,
					];

					$event = array_merge($event, $reg_infos);

					if ($linkid) {
						$eventsOfDay[$event['date_i']] = $event;
					}
				}

				unset($d);
			}
		}

		ksort($eventsOfDay);

		return $eventsOfDay;
	}


	/**
	 * Single Dates list for one event
	 */
	public static function getDatelist($dates)
	{
		$dates = iCString::isSerialized($dates) ? unserialize($dates) : [];
		$list  = [];

		foreach ($dates as &$d) {
			if (iCDate::isDate($d)) {
				$list[]= date('Y-m-d H:i', strtotime($d));
			}
		}

		return $list;
	}

	/**
	 * Function to get Format Date (using option format, and translation)
	 */
	public static function formatDate($date, $format, $separator, $tz = null)
	{
		// Component Params
		$iCparams = ComponentHelper::getParams('com_icagenda');

		// Date Format Option (Global Component Option)
		$date_format_global = $iCparams->get('date_format_global', 'Y - m - d');

		// Set Date Format option to be used
		$format = $format ?: $date_format_global;

		$day_of_the_week = '';

		if ((int) $format) {
			$day_of_the_week = '<span class="ic-dialog-day-of-week">' . Text::_(strtoupper(date('l', strtotime($date)))) . '</span>';

			if ($format == 7) {
				$format = 2;
			}
			if ($format == 8 || $format == 12) {
				$format = 4;
			}
			if ($format == 10) {
				$format = 5;
			}
			if ($format == 11) {
				$format = 3;
			}
		}

		// Date Separator Option (Global Component Option)
		$date_separator_global = $iCparams->get('date_separator', ' ');

		// Set Date Separator option to be used
		$separator = $separator ?: $date_separator_global;

		if (!is_numeric($format)) {
			// Update old Date Format options of versions before 2.1.7
			$format = str_replace(array('nosep', 'nosep', 'sepb', 'sepa'), '', $format);
			$format = str_replace('.', ' .', $format);
			$format = str_replace(',', ' ,', $format);
		}

		$dateFormatted = iCGlobalize::dateFormat($date, $format, $separator, $tz);

		return $day_of_the_week . $dateFormatted;
	}

	/**
	 * Get the class name for WCAG 2.0 text color depending on the background color.
	 * NOTE: The number of the day is large text; a contrast of 3 is therefore sufficient
	 *       for WCAG AA compliance.
	 *
	 * @param   string  $hex_color  The hexadecimal background color to test.
	 *
	 * @return  string  The text color class name (.ic-text-white or .ic-text-black).
	 */
	public static function getTextColorClass(string $hex_color, $a11y = 0): string
	{
		$white = '#ffffff';

		if (empty($hex_color)) {
			return $white;
		}

		if ($a11y == 1) {
			$ratio = '3';
		} elseif ($a11y == 2) {
			$ratio = '4.5';
		} else {
			$ratio = '4.5';
		}
		// Get contrast ratio white on background color.
		$contrast_ratio = self::calculateRatio($white, $hex_color);

		// If contrast ratio is enough, return white color.
		if ($contrast_ratio >= $ratio) {
			return ' ic-text-white';
		}

		return ' ic-text-black';
	}

	/**
	 * Calculate the contrast ratio between a foreground color and a bakcground color.
	 *
	 * @param   string  $fg  The foreground color
	 * @param   string  $bg  The background color
	 *
	 * @return  string  The contrast ratio
	 */
	protected static function calculateRatio($fg = null, $bg = null)
	{
		if (! $fg || ! $bg) {
			return;
		}

		$fgLuminance = static::get_luminance($fg);
		$bgLuminance = static::get_luminance($bg);

		return round((max($fgLuminance, $bgLuminance) + 0.05) / (min($fgLuminance, $bgLuminance) + 0.05) * 10) / 10;
	}

	/**
	 * Calculate luminance of a color.
	 *
	 * @see     https://github.com/breadthe/php-contrast/blob/master/src/HexColorPair.php#L92-L112
	 *
	 * @param   string  $color  Hex color with 6 or 8 digits
	 *
	 * @return  float   Color luminance
	 */
	public static function get_luminance(string $color): float
	{
		// Get decimal values
		$red   = \base_convert( \substr( $color, 1, 2 ), 16, 10 );
		$green = \base_convert( \substr( $color, 3, 2 ), 16, 10 );
		$blue  = \base_convert( \substr( $color, 5, 2 ), 16, 10 );

		// Get sRGB values
		$red_srgb   = $red / 255;
		$green_srgb = $green / 255;
		$blue_srgb  = $blue / 255;

		// Calculate luminance
		$r = ( $red_srgb <= .03928 ) ? $red_srgb / 12.92 : \pow( ( ( $red_srgb + .055 ) / 1.055 ), 2.4 );
		$g = ( $green_srgb <= .03928 ) ? $green_srgb / 12.92 : \pow( ( ( $green_srgb + .055 ) / 1.055 ), 2.4 );
		$b = ( $blue_srgb <= .03928 ) ? $blue_srgb / 12.92 : \pow( ( ( $blue_srgb + .055 ) / 1.055 ), 2.4 );

		return .2126 * $r + .7152 * $g + .0722 * $b;
	}
}
