<?php
/**
 * iCalcreator, the PHP class package managing iCal (rfc2445/rfc5445) calendar information.
 *
 * This file is a part of iCalcreator.
 *
 * @author    Kjell-Inge Gustafsson, kigkonsult <ical@kigkonsult.se>
 * @copyright 2007-2024 Kjell-Inge Gustafsson, kigkonsult AB, All rights reserved
 * @link      https://kigkonsult.se
 * @license   Subject matter of licence is the software iCalcreator.
 *            The above copyright, link, package and version notices,
 *            this licence notice and the invariant [rfc5545] PRODID result use
 *            as implemented and invoked in iCalcreator shall be included in
 *            all copies or substantial portions of the iCalcreator.
 *
 *            iCalcreator is free software: you can redistribute it and/or modify
 *            it under the terms of the GNU Lesser General Public License as
 *            published by the Free Software Foundation, either version 3 of
 *            the License, or (at your option) any later version.
 *
 *            iCalcreator is distributed in the hope that it will be useful,
 *            but WITHOUT ANY WARRANTY; without even the implied warranty of
 *            MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *            GNU Lesser General Public License for more details.
 *
 *            You should have received a copy of the GNU Lesser General Public License
 *            along with iCalcreator. If not, see <https://www.gnu.org/licenses/>.
 */
declare( strict_types = 1 );
namespace Kigkonsult\Icalcreator\Traits;

use Kigkonsult\Icalcreator\Formatter\Property\SingleProps;
use Kigkonsult\Icalcreator\Pc;
use Kigkonsult\Icalcreator\Util\Util;

/**
 * SUMMARY property functions
 *
 * @since 2.41.85 2024-01-18
 */
trait SUMMARYtrait
{
    /**
     * @var null|Pc component property SUMMARY value
     */
    protected ? Pc $summary = null;

    /**
     * Return formatted output for calendar component property summary
     *
     * @return string
     */
    public function createSummary() : string
    {
        return SingleProps::format(
            self::SUMMARY,
            $this->summary,
            $this->getConfig( self::ALLOWEMPTY ),
            $this->getConfig( self::LANGUAGE )
        );
    }

    /**
     * Delete calendar component property summary
     *
     * @return bool
     * @since  2.27.1 - 2018-12-15
     */
    public function deleteSummary() : bool
    {
        $this->summary = null;
        return true;
    }

    /**
     * Get calendar component property summary
     *
     * @param null|bool   $inclParam
     * @return bool|string|Pc
     * @since 2.41.85 2024-01-18
     */
    public function getSummary( ? bool $inclParam = false ) : bool | string | Pc
    {
        if( empty( $this->summary )) {
            return false;
        }
        return $inclParam ? clone $this->summary : $this->summary->getValue();
    }

    /**
     * Return bool true if set (and ignore empty property)
     *
     * @return bool
     * @since 2.41.88 2024-01-17
     */
    public function isSummarySet() : bool
    {
        return self::isPropSet( $this->summary );
    }

    /**
     * Set calendar component property summary
     *
     * @param null|string|Pc   $value
     * @param null|mixed[] $params
     * @return static
     * @since 2.41.85 2024-01-18
     */
    public function setSummary( null|string|Pc $value = null, ? array $params = [] ) : static
    {
        $pc      = Pc::factory( $value, $params );
        $pcValue = $pc->getValue();
        if( empty( $pcValue )) {
            $this->assertEmptyValue( $pcValue, self::SUMMARY );
            $pc->setEmpty();
        }
        else {
            $pc->setValue( Util::assertString( $pcValue, self::SUMMARY ));
        }
        $this->summary = $pc;
        return $this;
    }
}
